"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableConfig = void 0;
var cells_class_1 = require("./cell/cells.class");
var config_multicoord_1 = require("./config/config-multicoord");
var preferences_class_1 = require("./config/preferences.class");
/**
 * Table Config API
 */
var TableConfig = /** @class */ (function () {
    /**
     * Create TableConfig instance
     */
    function TableConfig() {
        this._cells = new cells_class_1.Cells();
        Object.defineProperty(this, '_cells', { enumerable: false, configurable: false, writable: false, value: this._cells });
    }
    /**
     * Get cell config
     * @param x 0-based coordinate index (-1 for row header)
     * @param y 0 based coordinate index (-1 for column header)
     * @returns cell config
     */
    TableConfig.prototype.getCellConfig = function (x, y) {
        return this._cells.configs.getsert(x, y).value;
    };
    /**
     * Get meshed (cell > column > row > table) configuration for cell
     * @param x 0-based coordinate index (-1 for row header)
     * @param y 0 based coordinate index (-1 for column header)
     * @returns cell render config
     */
    TableConfig.prototype.getCellRenderConfig = function (x, y) {
        return this._cells.configs.getsert(x, y).renderConfig;
    };
    /**
     * Get column config
     * @param x 0-based coordinate index (-1 for row header)
     * @returns column config
     */
    TableConfig.prototype.getColumnConfig = function (x) {
        return this._cells.configs.getColumn(x).value;
    };
    /**
     * Get generic table preferences
     * @returns preferences object
     */
    TableConfig.prototype.getPreferences = function () {
        return this._cells.configs.preferences.value;
    };
    /**
     * Get row config
     * @param y 0 based coordinate index (-1 for column header)
     * @returns row config
     */
    TableConfig.prototype.getRowConfig = function (y) {
        return this._cells.configs.getRow(y).value;
    };
    /**
     * Get table config
     * @returns table config
     */
    TableConfig.prototype.getTableConfig = function () {
        return this._cells.configs.table.value;
    };
    /**
     * Update cell config
     * @param x 0-based coordinate index (-1 for row header) or pattern like: '0..2,5' (from..to, to included)
     * @param y 0 based coordinate index (-1 for column header) or pattern like: '0..2,5' (from..to, to included)
     * @param config updates
     * @returns this Table
     */
    TableConfig.prototype.setCellConfig = function (x, y, config) {
        var xs = (0, config_multicoord_1.configMulticoord)(x);
        var ys = (0, config_multicoord_1.configMulticoord)(y);
        for (var _i = 0, xs_1 = xs; _i < xs_1.length; _i++) {
            var _x = xs_1[_i];
            for (var _a = 0, ys_1 = ys; _a < ys_1.length; _a++) {
                var _y = ys_1[_a];
                this._cells.configs.upsert(_x, _y, config);
            }
        }
        return this;
    };
    /**
     * Update column config
     * @param x 0-based coordinate index (-1 for row header) or pattern like: '0..2,5' (from..to, to included)
     * @param config updates
     * @returns this Table
     */
    TableConfig.prototype.setColumnConfig = function (x, config) {
        var xs = (0, config_multicoord_1.configMulticoord)(x);
        for (var _i = 0, xs_2 = xs; _i < xs_2.length; _i++) {
            var _x = xs_2[_i];
            this._cells.configs.setColumn(_x, config);
        }
        return this;
    };
    /**
     * Update generic table preferences
     * @param preferences updates
     * @returns this Table
     */
    TableConfig.prototype.setPreferences = function (preferences) {
        this._cells.configs.preferences.value = preferences;
        return this;
    };
    /**
     * Update row config
     * @param y 0 based coordinate index (-1 for column header) or pattern like: '0..2,5' (from..to, to included)
     * @param config updates
     * @returns this Table
     */
    TableConfig.prototype.setRowConfig = function (y, config) {
        var ys = (0, config_multicoord_1.configMulticoord)(y);
        for (var _i = 0, ys_2 = ys; _i < ys_2.length; _i++) {
            var _y = ys_2[_i];
            this._cells.configs.setRow(_y, config);
        }
        return this;
    };
    /**
     * Update table config
     * @param config updates
     * @returns this Table
     */
    TableConfig.prototype.setTableConfig = function (config) {
        this._cells.configs.table.value = config;
        return this;
    };
    TableConfig._preferences = new preferences_class_1.Preferences();
    return TableConfig;
}());
exports.TableConfig = TableConfig;
