import { AnalyzeQuery, AnalyzeResults, Document as IDocument, Index as IndexSchema, IndexDocument, IndexingResult, IndexingResults, IndexStatistics, Query, SearchDocument, SearchResults, SuggestQuery, SuggestResults } from 'azure-search-types';
import { SearchRequester } from "../search-requester";
import { SearchResource } from "../search-resource";
import { AzureSearchResponse, SearchCallback, SearchOptions } from '../types';
import { FieldName, QueryBuilder, QueryFacet } from "./builders";
import { IndexStream } from './index-stream';
export { IndexSchema, Query, SuggestQuery, AnalyzeQuery, IndexDocument, SearchResults, SuggestResults, AnalyzeResults, IndexingResult, IndexingResults, IndexStatistics, IDocument, SearchDocument, };
export interface DocumentParseOptions {
    parseDates?: boolean;
}
export interface SearchResponse<T> extends AzureSearchResponse<SearchResults<T>> {
}
export interface SuggestResponse<T> extends AzureSearchResponse<SuggestResults<T>> {
}
export interface SearchResponse<T> extends AzureSearchResponse<SearchResults<T>> {
}
export declare type OptionsOrBatchIndexedCallback = SearchOptions | IndexedCallback;
export declare type IndexedCallback = (err?: Error, count?: number) => PromiseLike<void>;
export declare type BatchIndexed = (results?: IndexingResult[]) => PromiseLike<void>;
export * from 'azure-search-types/dist/indexes/search';
export declare class SearchIndex<TDocument = any> extends SearchResource<IndexSchema> {
    /**
     * Manage Azure Search index resources
     * @param requester http handler
     * @param type must be 'indexes'
     * @param name the name of the current search index
     */
    constructor(requester: SearchRequester, type: string, name: string);
    buildQuery(): QueryBuilder<TDocument>;
    /**
     * Execute a search query
     * @param query query to execute
     * @param options optional request options
     */
    search(query?: Query, options?: SearchOptions & DocumentParseOptions): Promise<SearchResponse<TDocument>>;
    search(query: Query, callback: SearchCallback<SearchResults<TDocument>>): void;
    search(query: Query, options: SearchOptions & DocumentParseOptions, callback: SearchCallback<SearchResults<TDocument>>): void;
    /**
     * Execute a suggest query
     * @param query query to execute
     * @param options optional request options
     */
    suggest(query: SuggestQuery, options?: SearchOptions & DocumentParseOptions): Promise<SuggestResponse<TDocument>>;
    suggest(query: SuggestQuery, callback: SearchCallback<SuggestResults<TDocument>>): void;
    suggest(query: SuggestQuery, options: SearchOptions & DocumentParseOptions, callback: SearchCallback<SuggestResults<TDocument>>): void;
    /**
     * Perform indexing analysis on text
     * @param query query to execute
     * @param options optional request options
     */
    analyze(query: AnalyzeQuery, options?: SearchOptions): Promise<AzureSearchResponse<AnalyzeResults>>;
    analyze(query: AnalyzeQuery, callback: SearchCallback<AnalyzeResults>): void;
    analyze(query: AnalyzeQuery, options: SearchOptions, callback: SearchCallback<AnalyzeResults>): void;
    /**
     * Add, remove, or update documents in the search index. This function handles batching of content to fit within indexing request limits.
     * @param documents documents to index
     * @param options optional request options
     */
    index(documents: Array<IndexDocument & TDocument>, batchIndexed?: BatchIndexed, options?: SearchOptions): Promise<number>;
    index(documents: Array<IndexDocument & TDocument>, callback: IndexedCallback): void;
    index(documents: Array<IndexDocument & TDocument>, batchIndexed: BatchIndexed, callback: IndexedCallback): void;
    index(documents: Array<IndexDocument & TDocument>, batchIndexed: BatchIndexed, options: SearchOptions, callback: IndexedCallback): void;
    /**
     * Create a indexing stream suitable for piping in document objects.
     * Emits a 'data' event for every batch of documents that is indexed.
     * @param batchIndexed optional callback when a batch has been indexed
     * @param options optional request options
     */
    createIndexingStream(options?: SearchOptions): IndexStream;
    /**
     * Get document count and usage for the current index
     * @param options optional request options
     */
    statistics(options?: SearchOptions): Promise<AzureSearchResponse<IndexStatistics>>;
    statistics(callback: SearchCallback<IndexStatistics>): void;
    statistics(options: SearchOptions, callback: SearchCallback<IndexStatistics>): void;
    /**
     * Retrieve a single document from the current index
     * @param key document key
     * @param options optional request options
     */
    lookup(key: string, options?: SearchOptions): Promise<AzureSearchResponse<IDocument & TDocument>>;
    lookup(key: string, callback: SearchCallback<IDocument & TDocument>): void;
    lookup(key: string, options: SearchOptions, callback: SearchCallback<IDocument & TDocument>): void;
    /**
     * Retrieve a count of the number of documents in a search index
     * @param options optional request options
     */
    count(options?: SearchOptions): Promise<AzureSearchResponse<number>>;
    count(callback: SearchCallback<number>): void;
    count(options: SearchOptions, callback: SearchCallback<number>): void;
    /**
     * Create a new FacetBuilder for this index
     */
    facet(field: FieldName<TDocument>): QueryFacet<TDocument>;
}
